#!/usr/bin/perl

# Ballot I/O functions

require "condorcet.pl";		# for dumping the matrices

{

package ballot_io;

sub DumpCondorcetRow {
	# dumps condorcet row from an array to the relevant condorcet array
	# $_[0] = number of candidates
	# $_[1] = current y position
	# $_[2] = reference to output array
	# $_[3] = reference to input

	my $numcand = shift;
	my $curypos = shift;
	my $outarrayref = shift;
	my $inarrayref = shift;

	for(my $x = 0; $x < $numcand; $x++) {
		$$outarrayref[$x][$curypos] = $$inarrayref[$x];
	}
}

sub GetCandidates {
	my $candidates_file = shift;
	my $candarray = shift;		# reference to array to store candidate
					# names in
	my $candlinkarray = shift;	# same, but with candidate links

	my $link;
	my $numcand = 0;

	open (CANDFILE, $candidates_file) or die "Cannot open candidate file: $!";

	while (<CANDFILE>) {
		if ( !( (/#.*/) || (/^ *\n/) ) ) {
			s/\n//g;
			push(@{$candarray}, $_);
			$link = <CANDFILE>;
			$link =~ s/\n//g;
			push(@{$candlinkarray}, $link),
			$numcand++;
		}
	}

	close (CANDFILE);

	return($numcand);
}


sub GetVotingData {
# GetCondorcetMatrix: Gets condorcet matrices (NPref and "real matrix") as well
# as acceptability list and number of voters, then dumps them into the 
# appropriate arrays.
# Only simple error checking is done, since the users won't have direct
# control of the files anyway.

# Only the real matrix would be required to perform an election, but
# the NP matrix is used for additional information, and the acceptability list
# for extended NOTA (NOTR).

# returns the number of voters.

	#$_[0] = reference to condorcet matrix
	#$_[1] = reference to condorcet NP matrix
	#$_[2] = reference to acceptability matrix
	#$_[3] = condorcet matrix file name
	#$_[4] = number of candidates

	my $counter;
	my @input;

	my $matrix = shift;
	my $NPMatrix = shift;
	my $acceptability = shift;
	my $condorcet_file = shift;
	my $numcand = shift;
	my $voters;

	open(CMFILE, $condorcet_file) or 
		die "Cannot open condorcet matrix file: $!";

	while (<CMFILE>) {
		# is it the condorcet matrix?
		if ($_ eq "CONDMAT\n") {
			#Yes it is, parse the rest
			for ($counter = 0; $counter < $numcand; $counter++) {
				$numberline = <CMFILE>;		# get a line
				$numberline =~ s/\n//g;		# erase newline

				# dump to condorcet array
				@input = split(/ /, $numberline);
				DumpCondorcetRow($numcand, $counter, $matrix, 
					\@input);
			}
		}

		# is it the NP matrix?
		if ($_ eq "CONDNPMAT\n") {
			#Yes it is, parse the rest (see CONDMAT)
			for ($counter = 0; $counter < $numcand; $counter++) {
				$numberline = <CMFILE>;
				$numberline =~ s/\n//g;
				
				@input = split(/ /, $numberline);
				DumpCondorcetRow($numcand, $counter,
					$NPMatrix, \@input);
			}
		}

		# is it the acceptability list?
		if ($_ eq "ACCEPT\n") {
			$numberline = <CMFILE>;
			$numberline =~s/\n//g;
			@{$acceptability} = split(/ /, $numberline);
		}

		# is it the number of voters?
		if ($_ eq "VOTERS\n") {
			$numberline = <CMFILE>;
			$numberline =~ s/\n//g;
			$voters = $numberline;
		}
	}

	close(CMFILE);

	return($voters);

}

sub DumpCondorcetMatrix {
	#Does the inverse of what GetCondorcetMatrix does.
	#Overwrites the condorcet matrix file.

	#$_[0] = reference to condorcet matrix
        #$_[1] = reference to condorcet NP matrix
        #$_[2] = reference to acceptability matrix
        #$_[3] = condorcet matrix file name
        #$_[4] = number of candidates
	#$_[5] = number of voters

	my $linear;	# this is the temporary variable we print for cond.
			# matrices

	my $matrix = shift;			# condorcet matrix ref
	my $NPMatrix = shift;			# NP matrix ref
	my $acceptability = shift;		# acceptability ref
	my $filename = shift;			# filename
	my $numcand = shift;			# number of candidates
	my $voters = shift;			# number of voters

	# First, open the file.

	open (CMFILE, ">$filename") or die "Cannot open condorcet matrix file: $!";
	flock CMFILE, 2;			# wait for exclusive access

	#Then start printing...

	#condorcet matrix
	print CMFILE "CONDMAT\n";      # condorcet matrix ID
	print CMFILE condorcet::PrintCondorcetMatrix($matrix, $numcand);

	#condorcet NP matrix
	print CMFILE "\nCONDNPMAT\n";  # condorcet NP matrix ID
	print CMFILE condorcet::PrintCondorcetMatrix($NPMatrix, $numcand);

	#acceptability array
	print CMFILE "\nACCEPT\n";
	foreach $i(@{$acceptability}) { print CMFILE "$i "; }

	#voters
	print CMFILE "\nVOTERS\n";
	print CMFILE "$voters\n";
	print CMFILE "\n";

	close(CMFILE);
}
	
;1
}
